<?php
/**
 * Plugin Name: Secure Login Shield
 * Plugin URI: https://plugins.bentreder.com/secure-login-shield
 * Description: Create a private login URL (e.g., /dragon-lair) and block direct access to /wp-login.php with a stealth 404. Logged-out visits to /wp-admin/ are redirected to the homepage. Only the private URL can be used to log in. Deactivate to fully revert.
 * Version: 1.3.0
 * Author: Ben Treder
 * Author URI: https://bentreder.com
 * Donate link: https://www.buymeacoffee.com/bentreder
 * License: GPL2
 * Text Domain: secure-login-shield
 */

if ( ! defined( 'ABSPATH' ) ) exit;

define( 'SLS_OPTION_KEY', 'sls_settings' );
define( 'SLS_DEFAULT_SLUG', 'wp-login.php' ); // default stays wp-login.php

/**
 * Get sanitized slug from options (or default).
 */
function sls_get_slug() {
    $opts = get_option( SLS_OPTION_KEY, [] );
    $slug = isset( $opts['slug'] ) ? trim( (string) $opts['slug'] ) : '';
    if ( $slug === '' ) {
        $slug = SLS_DEFAULT_SLUG;
    }
    $slug = sanitize_title_with_dashes( $slug );
    return $slug !== '' ? $slug : SLS_DEFAULT_SLUG;
}

/**
 * Add rewrite rule for /{slug} → wp-login.php.
 */
add_action( 'init', function() {
    $slug = sls_get_slug();
    add_rewrite_rule( '^' . preg_quote( $slug, '/' ) . '/?$', 'wp-login.php', 'top' );
}, 9 );

/**
 * Replace WordPress-generated login URLs with our slug.
 */
add_filter( 'site_url', function( $url, $path, $scheme, $blog_id ) {
    $slug = sls_get_slug();
    if ( $path === 'wp-login.php' || ( function_exists( 'str_starts_with' ) && str_starts_with( (string) $path, 'wp-login.php?action=' ) ) ) {
        return site_url( $slug, $scheme );
    }
    return $url;
}, 10, 4 );

/**
 * ENFORCE: Only the custom slug may show the login screen.
 * Any direct /wp-login.php request → 404 Not Found (stealth mode).
 */
add_action( 'login_init', function() {
    // Unsplash & parse the current request path safely.
    $raw_request_uri = isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : '';
    $request_path    = wp_parse_url( $raw_request_uri, PHP_URL_PATH );
    $request_path    = is_string( $request_path ) ? $request_path : '';

    $slug           = sls_get_slug();
    $slug_pattern   = '#/' . preg_quote( $slug, '#' ) . '/?$#';

    // If the requested path does NOT end with "/{slug}" → serve stealth 404.
    if ( ! preg_match( $slug_pattern, $request_path ) ) {
        status_header( 404 );
        nocache_headers();

        // Try to use theme's 404 template if available.
        if ( function_exists( 'get_query_template' ) ) {
            $template_404 = get_query_template( '404' );
            if ( $template_404 && file_exists( $template_404 ) ) {
                include $template_404; // phpcs:ignore WordPressVIPMinimum.Files.IncludingFile.IncludingFile
                exit;
            }
        }

        // Fallback simple 404.
        echo '<!doctype html><html><head><meta charset="utf-8"><title>404 Not Found</title></head><body><h1>Not Found</h1><p>The requested resource was not found on this server.</p></body></html>';
        exit;
    }
});

/**
 * Redirect logged-out visitors from /wp-admin/ to homepage.
 */
add_action( 'init', function() {
    if ( is_admin() && ! is_user_logged_in() && !( defined( 'DOING_AJAX' ) && DOING_AJAX ) ) {
        wp_safe_redirect( home_url( '/' ) );
        exit;
    }
}, 11 );

/**
 * Flush rewrite rules on activation/deactivation.
 */
register_activation_hook( __FILE__, function() {
    if ( ! get_option( SLS_OPTION_KEY ) ) {
        add_option( SLS_OPTION_KEY, [ 'slug' => SLS_DEFAULT_SLUG ] );
    }
    do_action( 'init' );
    flush_rewrite_rules();

    // Set flag for admin notice.
    set_transient( 'sls_show_notice', true, 30 );
});

register_deactivation_hook( __FILE__, function() {
    flush_rewrite_rules(); // restore defaults
});

/**
 * Show activation notice reminding user to save permalinks & clear cache.
 */
add_action( 'admin_notices', function() {
    if ( get_transient( 'sls_show_notice' ) ) {
        delete_transient( 'sls_show_notice' );
        echo '<div class="notice notice-warning is-dismissible">
            <p><strong>Secure Login Shield activated.</strong><br>
            Go to <a href="' . esc_url( admin_url( 'options-permalink.php' ) ) . '">Settings → Permalinks</a> and click <em>Save Changes</em> to refresh rewrite rules.<br>
            If you use caching or a CDN, clear cache before testing your new login URL.</p>
        </div>';
    }
});

/**
 * Settings Page
 */
add_action( 'admin_menu', function() {
    add_options_page(
        __( 'Secure Login Shield', 'secure-login-shield' ),
        __( 'Secure Login Shield', 'secure-login-shield' ),
        'manage_options',
        'secure-login-shield',
        'sls_render_settings_page'
    );
});

function sls_render_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    if ( isset( $_POST['sls_save'] ) ) {
        check_admin_referer( 'sls_save_settings' );

        $slug = isset( $_POST['sls_slug'] ) ? sanitize_title_with_dashes( wp_unslash( $_POST['sls_slug'] ) ) : '';
        if ( $slug === '' ) {
            $slug = SLS_DEFAULT_SLUG;
        }

        update_option( SLS_OPTION_KEY, [ 'slug' => $slug ] );

        do_action( 'init' );
        flush_rewrite_rules();

        echo '<div class="updated"><p><strong>' . esc_html__( 'Settings saved.', 'secure-login-shield' ) . '</strong> ' . esc_html__( 'Private login URL is now:', 'secure-login-shield' ) . ' <code>/' . esc_html( $slug ) . '</code></p></div>';

        // Optional: remind to save permalinks & clear cache after changing slug.
        echo '<div class="notice notice-info is-dismissible">
            <p>' . sprintf(
                /* translators: 1: Permalinks URL */
                esc_html__( 'Tip: After changing your slug, visit %1$s and click Save Changes, then clear your cache/CDN.', 'secure-login-shield' ),
                '<a href="' . esc_url( admin_url( 'options-permalink.php' ) ) . '">Settings → Permalinks</a>'
            ) . '</p>
        </div>';
    }

    $opts      = get_option( SLS_OPTION_KEY, [ 'slug' => SLS_DEFAULT_SLUG ] );
    $slug      = isset( $opts['slug'] ) ? (string) $opts['slug'] : SLS_DEFAULT_SLUG;
    $login_url = site_url( $slug ); // escape on output below.
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Secure Login Shield', 'secure-login-shield' ); ?></h1>
        <p><em><?php echo esc_html__( 'By', 'secure-login-shield' ); ?> <strong>Ben Treder</strong> — <a href="<?php echo esc_url( 'https://BenTreder.com' ); ?>" target="_blank" rel="noopener">BenTreder.com</a></em></p>

        <h2><?php esc_html_e( 'How it works', 'secure-login-shield' ); ?></h2>
        <ul style="list-style:disc;margin-left:1.25em;">
            <li><?php echo wp_kses_post( __( 'Defaults to the standard <code>/wp-login.php</code> until you change it here.', 'secure-login-shield' ) ); ?></li>
            <li><?php echo wp_kses_post( __( 'Once changed, only your custom slug works — direct hits to <code>/wp-login.php</code> return a stealth 404.', 'secure-login-shield' ) ); ?></li>
            <li><?php echo wp_kses_post( __( 'Logged-out visitors to <code>/wp-admin/</code> are redirected to the homepage.', 'secure-login-shield' ) ); ?></li>
            <li><?php esc_html_e( 'Deactivate the plugin to restore default WordPress behavior.', 'secure-login-shield' ); ?></li>
        </ul>

        <form method="post" action="">
            <?php wp_nonce_field( 'sls_save_settings' ); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="sls_slug"><?php esc_html_e( 'Private Login Slug', 'secure-login-shield' ); ?></label></th>
                    <td>
                        <input name="sls_slug" id="sls_slug" type="text" class="regular-text" value="<?php echo esc_attr( $slug ); ?>" placeholder="dragon-lair" />
                        <p class="description">
                            <?php esc_html_e( 'Private login URL:', 'secure-login-shield' ); ?>
                            <code><?php echo esc_html( trailingslashit( home_url( '/' ) ) ); ?><strong><?php echo esc_html( $slug ); ?></strong></code>
                        </p>
                        <p class="description"><?php esc_html_e( 'Only letters, numbers, and dashes are allowed.', 'secure-login-shield' ); ?></p>
                    </td>
                </tr>
            </table>

            <p class="submit">
                <button type="submit" name="sls_save" class="button button-primary"><?php esc_html_e( 'Save Changes', 'secure-login-shield' ); ?></button>
                <a class="button" href="<?php echo esc_url( $login_url ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Open Private Login', 'secure-login-shield' ); ?></a>
            </p>
        </form>
    </div>
    <?php
}
